import { __awaiter } from "tslib";
import { TagActionProxy } from ".";
import { ROTATE_DIRECTION, ZOOM_ACTION, } from "../types";
import { buildKrpanoAction, buildKrpanoTagSetterActions } from "../utils";
export class KrpanoActionProxy {
    constructor(krpanoRenderer, name = "ReactKrpanoActionProxy") {
        this.eventHandlers = [];
        /**
         * 小行星视角
         */
        this.littlePlanetIntro = false;
        this.krpanoRenderer = krpanoRenderer;
        this.name = name;
        this.tagAction = new TagActionProxy(krpanoRenderer);
    }
    /**
     * 执行 Javascript 函数
     * @param action 动作
     * @param nexttick 是否在下一个渲染帧后执行
     */
    call(action, nexttick = false) {
        var _a;
        const actionStr = nexttick ? `nexttick(${action})` : action;
        (_a = this.krpanoRenderer) === null || _a === void 0 ? void 0 : _a.call(actionStr);
    }
    set(name, ...params) {
        this.call(buildKrpanoAction("set", name, ...params));
    }
    /**
     * 动态添加标签
     * @param tag 标签
     * @param name 名称
     * @param attrs 属性
     * @param nextTick 等待下一帧执行
     */
    setTag(tag, name, attrs, nextTick = false) {
        return __awaiter(this, void 0, void 0, function* () {
            let nexttick = nextTick;
            yield this.tagAction.waitIncludeLoaded();
            this.call(buildKrpanoTagSetterActions(name ? `${tag}[${name}]` : tag, attrs), nexttick);
        });
    }
    get(name) {
        var _a;
        return (_a = this.krpanoRenderer) === null || _a === void 0 ? void 0 : _a.get(name);
    }
    /**
     * 删除场景
     * @param name 场景名称
     */
    removeScene(name) {
        if (this.get("scene") &&
            typeof this.get("scene").removeItem === "function") {
            this.get("scene").removeItem(name);
        }
    }
    /**
     * 加载场景
     * @param name 场景 name
     */
    loadScene(name) {
        this.call(buildKrpanoAction("loadscene", name, "null", "MERGE", "OPENBLEND(0.5, 0.0, 0.75, 0.05, linear)"));
    }
    /**
     * 旋转视图
     * @param direction 方位
     * @param degrees 旋转度数，默认为 10
     */
    rotateView(direction, degrees = 10) {
        let str = "";
        const view = this.get("view");
        switch (direction) {
            case ROTATE_DIRECTION.LEFT:
                str = `view.hlookat, ${(view.hlookat || 0) - degrees}`;
                break;
            case ROTATE_DIRECTION.RIGHT:
                str = `view.hlookat, ${(view.hlookat || 0) + degrees}`;
                break;
            case ROTATE_DIRECTION.UP:
                str = `view.vlookat, ${(view.vlookat || 0) - degrees}`;
                break;
            case ROTATE_DIRECTION.DOWN:
                str = `view.vlookat, ${(view.vlookat || 0) + degrees}`;
                break;
        }
        this.call(buildKrpanoAction("tween", str, 0.5));
    }
    /**
     * 缩放视图
     * @param action 动作
     * @param num 缩放大小
     */
    zoomView(action, num = 10) {
        const view = this.get("view");
        const targetFov = action === ZOOM_ACTION.IN ? -num : num;
        this.call(buildKrpanoAction("tween", "view.fov", (view.fov || 0) + targetFov, 1));
    }
    on(eventName, selector, handler) {
        this.eventHandlers.push({
            eventName: eventName.toLowerCase(),
            selector,
            handler,
        });
        return this;
    }
    off(eventName, selector, handler) {
        this.eventHandlers = this.eventHandlers.filter((e) => !(e.eventName === eventName.toLowerCase() &&
            e.selector === selector &&
            e.handler === handler));
    }
    fire(eventName, selector) {
        this.eventHandlers
            .filter((e) => e.eventName === eventName.toLowerCase() && e.selector === selector)
            .map(({ handler }) => handler(this));
    }
    bindEvents(selector, mapEventsToHandler) {
        Object.keys(mapEventsToHandler).map((eventName) => {
            const func = mapEventsToHandler[eventName];
            if (func) {
                this.on(eventName, selector, func);
            }
        });
    }
    unbindEvents(selector, mapEventsToHandler) {
        Object.keys(mapEventsToHandler).map((eventName) => {
            const func = mapEventsToHandler[eventName];
            if (func) {
                this.off(eventName, selector, func);
            }
        });
    }
    addHotspot(name, attrs) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.tagAction.waitIncludeLoaded();
            this.call(buildKrpanoAction("addhotspot", name), true);
            this.setTag("hotspot", name, attrs);
        });
    }
    removeHotspot(name) {
        this.call(buildKrpanoAction("removehotspot", name), true);
    }
    addLayer(name, attrs) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.tagAction.waitIncludeLoaded();
            this.call(buildKrpanoAction("addlayer", name), true);
            this.setTag("layer", name, attrs);
        });
    }
    removeLayer(name) {
        this.call(buildKrpanoAction("removelayer", name), true);
    }
}
